#include <d3d9.h>
#include <d3dx9.h>

#include <algorithm>

#include "HomeworkRendererFunctions.h"
#include "Utilities.h"

float const EPSILON = 1e-10f;

struct Edge2D
{
    float a, b, c;
    bool inclusive;
};


// CheckEdge - Verifies whether a point is on the positive side
// of an edge, and returns the barycentric coordinate for the
// opposing vertex.

inline
bool CheckEdge(float fx, float fy, Edge2D const& edge, float& d)
{
    d = edge.a * fx + edge.b * fy + edge.c;

    if (edge.inclusive)
    {
        return d >= 0;
    }
    else
    {
        return d > 0;
    }
}

unsigned int MyRendererFunctions::Rasterize(ScreenTriangle const& triangle, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize) const
{
    LightedVertex const* vtx = triangle.vtx;

    // Extract the triangle extents

    float fminX = vtx[0].position.x;
    float fminY = vtx[0].position.y;
    float fmaxX = vtx[0].position.x;
    float fmaxY = vtx[0].position.y;

    for (unsigned int i = 1; i < 3; ++i)
    {
        if (fminX > vtx[i].position.x) fminX = vtx[i].position.x;
        if (fminY > vtx[i].position.y) fminY = vtx[i].position.y;
        if (fmaxX < vtx[i].position.x) fmaxX = vtx[i].position.x;
        if (fmaxY < vtx[i].position.y) fmaxY = vtx[i].position.y;
    }

    // Looping using floats is a bad thing (because of precision issues),
    // so we use integers instead.

    unsigned int const minX = max((unsigned int) floor(fminX), params.scissorX);
    unsigned int const minY = max((unsigned int) floor(fminY), params.scissorY);
    unsigned int const maxX = min((unsigned int) ceil (fmaxX), params.scissorX + params.scissorWidth  - 1);
    unsigned int const maxY = min((unsigned int) ceil (fmaxY), params.scissorY + params.scissorHeight - 1);

    // Prepare the edge equations

    Edge2D edge[3];

    for (unsigned int i = 0; i < 3; ++i)
    {
        // Get the opposite edge's vertex indices.

        unsigned int i0 = i == 2? 0: i + 1;
        unsigned int i1 = i == 0? 2: i - 1;

        // Make the vertex ordering deterministic.
        // All triangles sharing an edge will get
        // the exact same numbers for that edge.

        if (vtx[i0].position.x > vtx[i1].position.x)
        {
            std::swap(i0, i1);
        }
        else if (vtx[i0].position.x == vtx[i1].position.x)
        {
            if (vtx[i0].position.y > vtx[i1].position.y)
            {
                std::swap(i0, i1);
            }
        }

        // Calculate the edge equation.

        float const a = vtx[i1].position.y - vtx[i0].position.y;
        float const b = vtx[i0].position.x - vtx[i1].position.x;
        float const c = -(vtx[i0].position.x * a + vtx[i0].position.y * b);

        // Make the positive side face the inside of the triangle.
        // Detect degenerate triangles.
        // And normalize the equation.

        float const d = vtx[i].position.x * a + vtx[i].position.y * b + c;

        if (fabs(d) <= EPSILON)
        {
            // Degenerate triangle.
            return 0;
        }

        edge[i].a = a/d;
        edge[i].b = b/d;
        edge[i].c = c/d;
        edge[i].inclusive = d >= 0;
    }

    // Vertex attributes divided by W (position.w already contains one-over-W).

    float const fogOverW[3] = { vtx[0].fog * vtx[0].position.w, vtx[1].fog * vtx[1].position.w, vtx[2].fog * vtx[2].position.w };

    VertexData const dataOverW[3] =
    {
        { vtx[0].data.diffuse * vtx[0].position.w, vtx[0].data.specular * vtx[0].position.w, vtx[0].data.texCoord * vtx[0].position.w },
        { vtx[1].data.diffuse * vtx[1].position.w, vtx[1].data.specular * vtx[1].position.w, vtx[1].data.texCoord * vtx[1].position.w },
        { vtx[2].data.diffuse * vtx[2].position.w, vtx[2].data.specular * vtx[2].position.w, vtx[2].data.texCoord * vtx[2].position.w },
    };

    // Finally, extract the pixels contained within the triangle.

    unsigned int resultCount = 0;

    for (unsigned int y = minY; y <= maxY; ++y)
    {
        // Add 0.5 to the coordinates so that we use the coordinates of the pixel center.

        float const fy = float(y) + 0.5f;

        for (unsigned int x = minX; x <= maxX; ++x)
        {
            float const fx = float(x) + 0.5f;

            float b0, b1, b2; // Barycentric coordinates.

            if (CheckEdge(fx, fy, edge[0], b0)
             && CheckEdge(fx, fy, edge[1], b1)
             && CheckEdge(fx, fy, edge[2], b2))
            {
                // Z and one-over-W are interpolated linearly.

                float const   z = b0 * vtx[0].position.z + b1 * vtx[1].position.z + b2 * vtx[2].position.z;
                float const oow = b0 * vtx[0].position.w + b1 * vtx[1].position.w + b2 * vtx[2].position.w;

                if (z >= 0 && fabs(oow) >= EPSILON)
                {
                    // Perspective correction of interpolated data requires dividing by w.
                    // We divide the barycentric coordinates here for efficiency.

                    float const w = 1.0f / oow;

                    float const b0w = b0 * w;
                    float const b1w = b1 * w;
                    float const b2w = b2 * w;

                    if (resultCount >= pixelsSize)
                    {
                        throw "Rasterize: insufficient space for the result";
                    }

                    Pixel& p = pixels[resultCount];
                    p.x             = x;
                    p.y             = y;
                    p.z             = z;
                    p.fog           = b0w * fogOverW[0]           + b1w * fogOverW[1]           + b2w * fogOverW[2];
                    p.data.diffuse  = b0w * dataOverW[0].diffuse  + b1w * dataOverW[1].diffuse  + b2w * dataOverW[2].diffuse;
                    p.data.specular = b0w * dataOverW[0].specular + b1w * dataOverW[1].specular + b2w * dataOverW[2].specular;
                    p.data.texCoord = b0w * dataOverW[0].texCoord + b1w * dataOverW[1].texCoord + b2w * dataOverW[2].texCoord;

                    ++resultCount;
                }
            }
        }
    }

    return resultCount;
}
